import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import seaborn as sns
import scipy.stats as stats
import pickle

atp_tennis = pd.read_csv('atp_tennis.csv')

# Convert the 'Date' column to datetime format
atp_tennis['Date'] = pd.to_datetime(atp_tennis['Date'])

# Create a new DataFrame with separate rows for each player
player_data = pd.DataFrame(columns=['Player', 'Rank', 'Win_Loss_Ratio', 'Surface', 'Date'])

# Add Player_1 data to the new DataFrame
player_data_1 = atp_tennis[['Player_1', 'Rank_1', 'Win_Loss_Ratio_1', 'Surface', 'Date']].copy()
player_data_1.columns = ['Player', 'Rank', 'Win_Loss_Ratio', 'Surface', 'Date']

# Add Player_2 data to the new DataFrame
player_data_2 = atp_tennis[['Player_2', 'Rank_2', 'Win_Loss_Ratio_2', 'Surface', 'Date']].copy()
player_data_2.columns = ['Player', 'Rank', 'Win_Loss_Ratio', 'Surface', 'Date']

# Concatenate Player_1 and Player_2 data
player_data = pd.concat([player_data_1, player_data_2], ignore_index=True)

# Group the DataFrame by year and surface type, and compute the average win/loss ratio
player_data['Year'] = player_data['Date'].dt.year
yearly_surface_data = player_data.groupby(['Year', 'Surface']).agg({'Win_Loss_Ratio': 'mean'}).reset_index()

# Create a line chart comparing win/loss ratios across years and surface types
plt.figure(figsize=(12, 6))
sns.lineplot(x='Year', y='Win_Loss_Ratio', hue='Surface', data=yearly_surface_data)
plt.title('Win/Loss Ratios by Surface Type and Year')
plt.ylabel('Average Win/Loss Ratio')
plt.savefig("./ref_result/lineplot.png")
plt.show()

# Perform a linear regression for each surface type to analyze trends
trend_results = {}
for surface in ['Hard', 'Clay', 'Grass']:
    surface_data = yearly_surface_data[yearly_surface_data['Surface'] == surface]
    slope, intercept, r_value, p_value, std_err = stats.linregress(surface_data['Year'], surface_data['Win_Loss_Ratio'])
    trend_results[surface] = {'Slope': slope, 'P-value': p_value}

print(trend_results)
pickle.dump(trend_results,open("./ref_result/trend_results.pkl","wb"))